package gov.va.med.mhv.bluebutton.bbmi.imaging.util;

import gov.va.med.mhv.bluebutton.model.Study;
import gov.va.med.mhv.bluebutton.model.StudyJob;
import gov.va.med.mhv.bluebutton.repository.StudyJobRepository;
import gov.va.med.mhv.bluebutton.repository.StudyRepository;
import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.core.crypto.MHVCipher;
import gov.va.med.mhv.core.crypto.MHVCipherOutputStream;
import gov.va.med.mhv.integration.phr.client.cvix.Study.Serieses.Series;
import gov.va.med.mhv.integration.phr.client.cvix.Study.Serieses.Series.Images.Image;
import gov.va.med.mhv.integration.phr.service.cvix.client.CVIXClient;
import gov.va.med.mhv.integration.phr.service.cvix.client.CVIXClientException;
import gov.va.med.mhv.integration.phr.service.cvix.model.ImageTO;
import gov.va.med.mhv.usermgmt.common.enums.ActivityActionTypeEnumeration;
import gov.va.med.mhv.usermgmt.common.enums.ActivityTypeEnumeration;
import gov.va.med.mhv.usermgmt.service.AccountActivityCreatorService;
import gov.va.med.mhv.usermgmt.service.PatientWebService;
import gov.va.med.mhv.usermgmt.util.activity.ActivityHelper;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.annotation.Resource;
import javax.ws.rs.WebApplicationException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dcm4che3.media.RecordFactory;
import org.dcm4che3.tool.dcmdir.DcmDir;
import org.springframework.beans.factory.annotation.Value;

public class EncryptedStudy {

    static final Set<PosixFilePermission> POSIX_750 = PosixFilePermissions.fromString("rwxr-x---");
	
	@Resource
	private StudyJobRepository studyJobRepository;

	private static final Log log = LogFactory.getLog(EncryptedStudy.class);

	private static byte[] CACHED_RECORD_FACTORY = null;

	@Resource
	private CVIXClient cvixService;

	@Resource
	private PatientWebService patientProxy;

	@Resource
	private StudyRepository studyRespository;

	@Resource(name = "activityProxy")
	private AccountActivityCreatorService activityProxy;

	private long patientId;
	private String localFolder;
	
	private String icn;
	private String studyId;
	private String serverNode;
	private Long userProfileId;

	@Value("${folder}")
	private String FOLDER_PROP;

	
	public EncryptedStudy() {
		super();
	}

	private StudyJob updateStatus(String percent, StudyJob job) throws Exception {
		try {
//			System.out.println(">>>Updating status of " + job.getStudyIdUrn() + " to " + percent + "%");
			job.setStatusText(percent);
			return saveJob(job);
		}
		catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	private StudyJob saveJob(StudyJob job) {
		job = studyJobRepository.save(job);
		return job;
	}
	

	private void updateStatusToError(StudyJob job) {
		try {
			//RECORD AAL ENTRY FOR ERROR TO PROCESS STUDY
			List<Study> studies = studyRespository.findByPatientIdAndStudyIdUrn(patientId, job.getStudyIdUrn());
			Study studyEntity = studies.get(0);
			SimpleDateFormat sdfDetail = new SimpleDateFormat("dd MMM yyyy @ HHmm");
			String detail = studyEntity.getProcedureName() + " performed on " + sdfDetail.format(studyEntity.getPerformedDate().getDatePrecise());
			activityProxy.createAccountActivityLog(ActivityHelper.createActivityDTOForSelf(userProfileId, false, ActivityTypeEnumeration.DOWNLOAD, ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_REQUEST_PROCESSED, detail));

			job.setStatus(StudyJobConstants.ERROR);
			job = saveJob(job);
		} catch (Exception e) {
			log.error("Exception setting error status: " + e.getMessage());
		}
		//Note: PLSQL call to notify user that download is NOT complete..  procedure will check whether use has enrolled etc.
		studyJobRepository.bbmiDownloadReminder(new Long(job.getPatientId()),new Long(job.getId()));	
	}

	public void writeStudy(StudyJob job) throws IOException, MHVException {
		DcmDir dcmDir = new DcmDir();
    	
		int totalItems = 0;

		if( this.userProfileId == null ) {
			this.userProfileId = patientProxy.getPatientByPatientId(patientId).getUserProfileId();
		}

		gov.va.med.mhv.integration.phr.client.cvix.Study study = null;
		String report = null;
		try {
			// GET STUDY
			study = cvixService.getStudyResult(icn, studyId);
			// GET REPORT
			report = cvixService.getReportResult(icn, studyId);
		} catch (CVIXClientException e2) {
			updateStatusToError(job);
			e2.printStackTrace();
			return;
		}

		totalItems = study.getImageCount().intValue() + 1; // Plus one for the report

		SimpleDateFormat sdf = new SimpleDateFormat("ddMMMyyyy");

		// CREATE ZIP FILENAME
		String procedureDateStr = "UNKNOWN";
		try {
			procedureDateStr = sdf.format(CVIXDateUtil.parseDate(study.getProcedureDate().toString()));
			procedureDateStr = procedureDateStr.toUpperCase();
		} catch (Exception e) {
			updateStatusToError(job);
			log.warn("Found an invalid date for procedureDate: '" + study.getProcedureDate() + "' for ICN: '" + icn + "', for StudyId: '" + studyId + "'");
			return;
		}

		// SET THE ZIP FILE NAME FOR CONTAINER [VA_IMG_{lastname}_{studyname}_{DDMMMYYYY}.zip]
		String filename = "VA_IMG_"
				+ (study.getPatientName()).substring(0, study.getPatientName().indexOf(','))
				+ "_"
				+ (study.getDescription().replaceAll(" ", "_")).replaceAll(",", "_")
				+ "_"
				+ procedureDateStr
				+ ".zip";

		List<Series> list = study.getSerieses().getSeries();

        File subFolder = new File(FOLDER_PROP+"/"+serverNode);
        subFolder.mkdirs();
        setPermissionsOnFile(subFolder.getPath());
        
        //Create temp file for the dicom since the dcm4che needs to write to a physical file
        File tempDICOMDIRFile = File.createTempFile(String.valueOf(patientId)+'_'+studyId, ".dd", new File(FOLDER_PROP+"/"+serverNode));
        setPermissionsOnFile(tempDICOMDIRFile.getPath());
        
        try {
			setupDicomDirFile(dcmDir, tempDICOMDIRFile);
		} catch (Exception e2) {
			throw new WebApplicationException(e2, 500);
		}
        
		File tempFile = File.createTempFile(String.valueOf(patientId)+'_'+studyId, ".tmp", new File(FOLDER_PROP+"/"+serverNode));
		
		MHVCipher c = MHVCipher.createCipher(StudyJobConstants.PASSPHRASE, false);
		MHVCipherOutputStream eos = c.getMHVCipherOutputStream(new FileOutputStream(tempFile));
		ZipOutputStream zos = new ZipOutputStream(eos);

		try {
			job.setFileName(filename);
			job.setTempFileName(tempFile.getName());
			job.setStatus(StudyJobConstants.PROCESSING);
			job.setStatusText("0");
			job = saveJob(job);
		} catch (Exception e) {
			throw new WebApplicationException(e, 500);
		}

		try {
			// ADD STUDY REPORT
			if (log.isInfoEnabled()) {
				log.info("Adding Report File");
			}
			zos.putNextEntry(new ZipEntry(StudyJobConstants.REPORT_TXT_FILENAME));
			zos.write(report.getBytes());
			zos.closeEntry();
			zos.flush();
		} catch (Exception e) {
			throw new WebApplicationException(e, 500);
		}

		if (log.isDebugEnabled()) {
			log.debug("Percent complete: " + String.valueOf((int) (90 / totalItems)) + "%" + " Total items: " + totalItems);
		}
		try {
			job = updateStatus(String.valueOf((int) (90 / totalItems)), job);
		} catch (Exception e) {
			throw new WebApplicationException(e, 500);
		}

		int seriesCount = 1;
		int imgCnt = 1;
		for (gov.va.med.mhv.integration.phr.client.cvix.Study.Serieses.Series s : list) {
			if (log.isInfoEnabled()) {
				log.info("Adding Series " + (seriesCount++) + " of " + list.size());
			}

			// FETCH AND ADD EACH IMAGE
			String directory = "SERIES_" + s.getSeriesNumber();
			gov.va.med.mhv.integration.phr.service.cvix.model.ImageTO image = null;

			// GET REFERENCE QUALITY (IMAGE/JPEG with 70 quality)
			for (Image img : s.getImages().getImage()) {
				try {
					String urn = img.getDiagnosticImageUri();
					image = cvixService.getImage(urn);
					String fileName = "DICOM/" + directory + "/" + image.getName();
					if (log.isInfoEnabled()) {
						log.info("Adding " + fileName + "(size: " + image.getContent().length + " bytes)");
					}
					zos.putNextEntry(new ZipEntry(fileName));
					zos.write(image.getContent());
					zos.closeEntry();
					zos.flush();
					
					//Add the DCM file to the DICOMDIR directory file
					addDicomDirFile(dcmDir, fileName, image.getContent());
					
					job = updateStatus(String.valueOf((int)((++imgCnt) * 90 / (totalItems))), job);
					if (job == null) {
						log.error("Job was cancelled due to error");
						//TODO: Clean up job and reset it?
						return;
					}

				} catch (Exception e) {
					updateStatusToError(job);
					throw new WebApplicationException(e, 500);
				}
			}
		}
		//CLOSE DICOMDIR FILE AFTER ALL THE IMAGES HAVE BEEN ADDED 
		closeDicomDirFile(dcmDir);
		
		try {
			// ADD STUDY DICOMDIR FILE
//			if (log.isInfoEnabled()) {
//				log.info("Adding DICOMDIR File");
//			}
			zos.putNextEntry(new ZipEntry("DICOMDIR"));
			Path path = Paths.get(tempDICOMDIRFile.getPath());
			byte[] data = Files.readAllBytes(path);
			tempDICOMDIRFile.delete();
			zos.write(data);
			zos.closeEntry();
			zos.flush();
		} catch (Exception e) {
			throw new WebApplicationException(e, 500);
		}
		
		try {
			zos.finish();
			zos.flush();
			zos.close();

			//SET THE PERMISSIONS TO 750
			setPermissionsOnFile(tempFile.getPath());
			
			//RECORD AAL ENTRY FOR SUCCESSFUL PROCESS STUDY
			List<Study> studies = studyRespository.findByPatientIdAndStudyIdUrn(patientId, job.getStudyIdUrn());
			Study studyEntity = studies.get(0);
			SimpleDateFormat sdfDetail = new SimpleDateFormat("dd MMM yyyy @ HHmm");
			String detail = studyEntity.getProcedureName() + " performed on " + sdfDetail.format(studyEntity.getPerformedDate().getDatePrecise());
			activityProxy.createAccountActivityLog(ActivityHelper.createActivityDTOForSelf(userProfileId, true, ActivityTypeEnumeration.DOWNLOAD, ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_REQUEST_PROCESSED, detail));

			// UPDATE PHRSTUDYJOB
			try {
				job.setFileSize(eos.getByteCount());
				if( StudyJobConstants.PREVIEW_ENABLED ) {
					job = updateStatus("90", job);
				} else {
					job.setStatus(StudyJobConstants.COMPLETE);
					job = updateStatus("100",job);
				}
			} catch (Exception e1) {
				throw new WebApplicationException(e1, 500);
			}

		} catch (Exception e) {

			//RECORD AAL ENTRY FOR SUCCESSFUL PROCESS STUDY
			List<Study> studies = studyRespository.findByPatientIdAndStudyIdUrn(patientId, job.getStudyIdUrn());
			Study studyEntity = studies.get(0);
			SimpleDateFormat sdfDetail = new SimpleDateFormat("dd MMM yyyy @ HHmm");
			String detail = studyEntity.getProcedureName() + " performed on " + sdfDetail.format(studyEntity.getPerformedDate().getDatePrecise());
			try {
				activityProxy.createAccountActivityLog(ActivityHelper.createActivityDTOForSelf(userProfileId, false, ActivityTypeEnumeration.DOWNLOAD, ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_REQUEST_PROCESSED, detail));
			} catch (MHVException e1) {
			}

			throw new WebApplicationException(e, 500);
		}

		writeStudyPreview(study, tempFile.getAbsoluteFile(), new String(report), job);
	
		//Note: PLSQL call to notify user that download is complete..  procedure will check whether use has enrolled etc.
		studyJobRepository.bbmiDownloadReminder(new Long(job.getPatientId()),new Long(job.getId()));
	}

	public void writeStudyPreview(gov.va.med.mhv.integration.phr.client.cvix.Study study, File tempFileForZip, String report, StudyJob job) throws IOException {

		int totalItems = study.getImageCount().intValue();

		List<Series> list = study.getSerieses().getSeries();
		
		//Only proceed if the # images <= MAX (currently 50)
		if( !(totalItems > StudyJobConstants.MAX_PREVIEW_IMAGES) ) {
			File tempFile = new File(tempFileForZip.getAbsoluteFile()+StudyJobConstants.PREVIEW_EXTENSION);
			MHVCipher c = MHVCipher.createCipher(StudyJobConstants.PASSPHRASE, false);
			MHVCipherOutputStream eos = c.getMHVCipherOutputStream(new FileOutputStream(tempFile));
			ZipOutputStream zos = new ZipOutputStream(eos);
	
			//First write the report
			zos.putNextEntry(new ZipEntry("Report.txt"));
			zos.write(report.getBytes());
			zos.closeEntry();
			zos.flush();
	
			int totalCnt = 1;
			for (Series s : list) {
				// FETCH AND ADD EACH IMAGE
				ImageTO image = null;
	
				if(totalCnt>totalItems) {
					//Exceeded the limit of X images
					break;
				}
	
				// GET REFERENCE QUALITY (IMAGE/JPEG or IMAGE/GIF or IMAGE/PNG with 70 quality)
	//			 	<referenceImageUri>
	//				imageURN=urn:vaimage:660-21803-21802-101xxxxxxxxxx&imageQuality=70&contentType=application/dicom,image/j2k,application/dicom,image/x-targa,*/*
	//				</referenceImageUri>
	//				<thumbnailImageUri>
	//				imageURN=urn:vaimage:660-21803-21802-101xxxxxxxxxx&imageQuality=20&contentType=image/jpeg,image/x-targa,image/bmp,*/*
	//				</thumbnailImageUri>
				int imageCnt = 1;
				for (Image img : s.getImages().getImage()) {
					try {
						String urn = "";
	
						if( img.getReferenceImageUri().contains("image/jpeg") ) {
							urn = img.getReferenceImageUri();
						} else if( img.getThumbnailImageUri().contains("image/jpeg") ) {
							urn = img.getThumbnailImageUri();
						} else {
							continue;
						}
	
						//Only fetch if jpeg
						urn = urn.substring(0,urn.indexOf("contentType")) + "contentType=image/jpeg";
	
						if(log.isDebugEnabled()) {
							log.debug("Using: " + urn);
						}
	
						if( urn.length() == 0 ){
							continue;
						}
	
						image = cvixService.getImage(urn);
	
						//ASSUMES <50 images in preview
						String fileName = "Series " + (s.getSeriesNumber().intValue()<10?"0":"") +s.getSeriesNumber() + " - Image " + (imageCnt<10?"0":"") + imageCnt + ".jpg";
						if (log.isDebugEnabled()) {
							log.debug("Adding " + fileName + "(size: " + image.getContent().length + " bytes)");
						}
						zos.putNextEntry(new ZipEntry(fileName));
						zos.write(image.getContent());
						zos.closeEntry();
						zos.flush();
	
						try {
							job = updateStatus(String.valueOf( 90 + (int)  (10 * ((float) totalCnt/totalItems ))), job );
						} catch (Exception e) {
							throw new WebApplicationException(e, 500);
						}
						imageCnt++;
						totalCnt++;
					} catch (Exception e) {
						//JUST CONTINUE BECAUSE WE NEED TO LEAVE IMAGES AS BLANK IF THEY ARE NOT AVAILABLE
						//throw new WebApplicationException(e, 500);
					}
				}
			}
			zos.finish();
			zos.flush();
			zos.close();

			//SET THE PERMISSIONS TO 750
			setPermissionsOnFile(tempFile.getPath());
		}

		try {
			// UPDATE PHRSTUDYJOB
			try {
				job.setStatus(StudyJobConstants.COMPLETE);
				job.setStatusText("100");
				job = saveJob(job);
			} catch (Exception e1) {
				throw new WebApplicationException(e1, 500);
			}
		} catch (Exception e) {
			throw new WebApplicationException(e, 500);
		}
	}

	public StudyJobRepository getStudyJobRepository() {
		return studyJobRepository;
	}

	public void setStudyJobRepository(StudyJobRepository studyJobRepository) {
		this.studyJobRepository = studyJobRepository;
	}

	public CVIXClient getCvixService() {
		return cvixService;
	}

	public void setCvixService(CVIXClient cvixService) {
		this.cvixService = cvixService;
	}

	public long getPatientId() {
		return patientId;
	}

	public void setPatientId(long patientId) {
		this.patientId = patientId;
	}

	public String getLocalFolder() {
		return localFolder;
	}

	public void setLocalFolder(String localFolder) {
		this.localFolder = localFolder;
	}

	public String getIcn() {
		return icn;
	}

	public void setIcn(String icn) {
		this.icn = icn;
	}

	public String getStudyId() {
		return studyId;
	}

	public void setStudyId(String studyId) {
		this.studyId = studyId;
	}

	public String getServerNode() {
		return serverNode;
	}

	public void setServerNode(String serverNode) {
		this.serverNode = serverNode;
	}
	
	public static void setPermissionsOnFile(String path) {
		Path p = Paths.get(path);
		try {
	    	Files.setPosixFilePermissions(p, POSIX_750);
		} catch (IOException e) {
			log.error("Unable to set permissions to 750 on file '" + path +"'");
		}
	}

	public static byte[] inputStreamToByteArray(InputStream is) throws IOException { 
		ByteArrayOutputStream baos = new ByteArrayOutputStream(); 
		int reads = is.read(); 
		while(reads != -1){ 
			baos.write(reads); 
			reads = is.read(); 
		} 
		return baos.toByteArray(); 
	} 

	public synchronized InputStream fetchRecordFactory() throws IOException {
		if( CACHED_RECORD_FACTORY == null ) {
			InputStream in = Thread.currentThread().getContextClassLoader().getResourceAsStream("/dicom/MHVRecordFactory.xml");
			if( in == null ) { 
				in = Thread.currentThread().getContextClassLoader().getResourceAsStream("dicom/MHVRecordFactory.xml");
			}
			CACHED_RECORD_FACTORY = inputStreamToByteArray(in);
		}
		ByteArrayInputStream bais = new ByteArrayInputStream(CACHED_RECORD_FACTORY);
		return bais;
	}
	
	
	public void setupDicomDirFile(DcmDir dcmDir, File tempFile) throws Exception {
		dcmDir.create(tempFile);
		dcmDir.setCheckDuplicate(false);
		RecordFactory rf = new RecordFactory();
		rf.loadConfiguration(fetchRecordFactory());
		dcmDir.setRecordFactory(rf);
	}
	
	public void addDicomDirFile(DcmDir dcmDir, String name, byte[] dicomFile) {
		try {
			ByteArrayInputStream bais = new ByteArrayInputStream(dicomFile);
			dcmDir.addReferenceTo(new MhvFile(name), bais);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	public void closeDicomDirFile(DcmDir dcmDir){
		dcmDir.close();
	}
	
}

